#include "NpAsyncRequests.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"
#include "SignInSony.h"


using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

#define min(a,b) ((a)<(b)?(a):(b))

namespace UnityPlugin
{
#if NP_HAS_NP_REQUESTS


	NpAsyncRequests gRequests;

	int NpAsyncRequests::s_RequestRef = 0;

	PRX_EXPORT ErrorCode PrxNpRequestCheckPlus(SceUserServiceUserId userId, uint64_t features)
	{
		return gRequests.CheckPlus(userId, features);
	}

	PRX_EXPORT ErrorCode PrxNpRequestCheckPlusDefaultUser(uint64_t features)
	{
		return gRequests.CheckPlus(GetUserId(), features);
	}

	PRX_EXPORT ErrorCode PrxNpRequestNotifyPlusFeature(SceUserServiceUserId userId, uint64_t features)
	{
		return gRequests.NotifyPlusFeature(userId, features);
	}

	PRX_EXPORT ErrorCode PrxNpRequestNotifyPlusFeatureDefaultUser(uint64_t features)
	{
		return gRequests.NotifyPlusFeature(GetUserId(), features);
	}

	
	PRX_EXPORT ErrorCode PrxNpRequestGetAccountLanguage(const char *onlineid)
	{
		return gRequests.GetAccountLanguage(onlineid);
	}

	
	PRX_EXPORT ErrorCode PrxNpRequestGetParentalControlInfo(const char *onlineid)
	{
		return gRequests.GetParentalControlInfo(onlineid);
	}

	
	PRX_EXPORT ErrorCode PrxNpRequestCheckNpAvailability(const char *onlineid)
	{
		return gRequests.CheckNpAvailability(onlineid);
	}

	
	PRX_EXPORT ErrorCode PrxNpRequestSetGamePresenceOnline(const char *onlineid)
	{
		return gRequests.SetGamePresenceOnline(onlineid);
	}





	NpAsyncRequests::NpAsyncRequests()
	{
	}



	NpAsyncRequests::UnityNpRequest * NpAsyncRequests::CreateRequest()
	{
		UnityNpRequest *request = new UnityNpRequest();
		
		
		request->m_requestType = UnityNpRequest::TypeCheckPlus;

		SceNpCreateAsyncRequestParameter requestParam;
		memset(&requestParam, 0, sizeof(requestParam));
		requestParam.size = sizeof(requestParam);
		requestParam.cpuAffinityMask = SCE_KERNEL_CPUMASK_USER_ALL & ~0x3;	// all cores excluding 0 (update) and 1 (render)
		requestParam.threadPriority = SCE_KERNEL_PRIO_FIFO_DEFAULT;

		int requestid =  sceNpCreateAsyncRequest(&requestParam);
		request->m_requestId = requestid;
//		printf("CreateRequest request 0x%p id:0x%x\n", request, request->m_requestId);

		if (requestid < 0)
		{
			switch(request->m_requestId)
			{
				case SCE_NP_ERROR_INVALID_ARGUMENT:		// Argument is invalid
				case SCE_NP_ERROR_OUT_OF_MEMORY:		// Not enough free memory
				case SCE_NP_ERROR_INVALID_SIZE:			// pParam->size is invalid
				case SCE_NP_ERROR_REQUEST_MAX:			// Created more than 32 requests at one time.
				default:
					UNITY_TRACE("ERROR: AsyncPlusCheck::CreateRequest() - sceNpCreateAsyncRequest() returned: %x\n", requestid);
					break;
			}
			DeleteRequest(request);
			return NULL;
		}

		request->m_requestId = requestid;

		return request;
	}


	void NpAsyncRequests::DeleteRequest(NpAsyncRequests::UnityNpRequest *request)
	{
		if (request==NULL) return;
		if (request->m_requestId != -1) 
		{
//			printf("deleting request 0x%p id:0x%x\n", request, request->m_requestId);
			int retDelete = sceNpDeleteRequest(request->m_requestId);
			if(retDelete < 0)
			{
				switch(retDelete)
				{
					case SCE_NP_ERROR_INVALID_ARGUMENT:			// reqId is not a positive number (>0)
					case SCE_NP_ERROR_REQUEST_NOT_FOUND:		// Request specified for reqId does not exist
					default:
						UNITY_TRACE("ERROR: AsyncPlusCheck::DeleteRequest() - sceNpDeleteRequest() returned: 0x%x\n", retDelete);
						break;
				}
			}
		}
		delete(request);
	}



	void NpAsyncRequests::Update()
	{
		int requestResult = 0;

		auto it = m_RequestList.begin();
		while (it != m_RequestList.end())
		{
			NpAsyncRequests::UnityNpRequest *request = it->second;

			int retPoll = sceNpPollAsync(request->m_requestId, &requestResult);
			if (retPoll == SCE_NP_POLL_ASYNC_RET_RUNNING) 
			{
				++it;
				continue;
			}			
			if (retPoll == SCE_NP_POLL_ASYNC_RET_FINISHED)
			{
//				UNITY_TRACE("ERROR: NpAsyncRequests::Update() - SCE_NP_POLL_ASYNC_RET_FINISHED returned: %x\n", requestResult);

				UnityPlugin::Messages::PluginMessage* msg = new UnityPlugin::Messages::PluginMessage();
				switch(request->m_requestType)
				{
				case UnityNpRequest::TypeCheckPlus:
					msg->type = Messages::kNPToolKit_CheckPlusResult;
					msg->SetData(request, sizeof(NpAsyncRequests::UnityNpRequest) );	// make a copy of the data into the message
					UnityPlugin::Messages::AddMessage(msg);
					break;
				case UnityNpRequest::TypeAccountLanguage:
					msg->type = Messages::kNPToolKit_AccountLanguageResult;
					msg->SetData(request, sizeof(NpAsyncRequests::UnityNpRequest) );	// make a copy of the data into the message
					UnityPlugin::Messages::AddMessage(msg);
					break;
				case UnityNpRequest::TypeParentalControlInfo:
					msg->type = Messages::kNPToolKit_ParentalControlResult;
					msg->SetData(request, sizeof(NpAsyncRequests::UnityNpRequest) );	// make a copy of the data into the message
					UnityPlugin::Messages::AddMessage(msg);
					break;

				case UnityNpRequest::TypeNPAvailabilty:
				case UnityNpRequest::TypeSetGamePresenceOnline:

				default:
					break;
				}

			}
			else
			{
				UNITY_TRACE("ERROR: NpAsyncRequests::Update() sceNpPollAsync() returned: %x:%x\n", retPoll, requestResult);
			}

			DeleteRequest(request);
			m_RequestList.erase(it++);
		}


	}


	ErrorCode NpAsyncRequests::GetAccountLanguage(const char *onlineId)
	{
		ResultCode Result("");
		if(onlineId ==  NULL)
		{
			UNITY_TRACE("NOTE: GetAccountLanguage() - missing onlne id\n");
			return NP_ERR_FAILED;
		}

		NpAsyncRequests::UnityNpRequest * request = CreateRequest();
		if(request ==  NULL)
		{
			UNITY_TRACE("NOTE: GetAccountLanguage() - Failed to create asynchronous request\n");
			return NP_ERR_FAILED;
		}
		m_RequestList.insert(std::pair<int,UnityNpRequest *>(request->m_requestId, request) );
		request->m_requestType = UnityNpRequest::TypeAccountLanguage;

		memcpy(&request->m_ALNpOnlineId , onlineId, min(strlen(onlineId)+1, sizeof(SceNpOnlineId)));
		request->m_LanguageCode.code[0] = 0x0;

		int retCheck = sceNpGetAccountLanguage(request->m_requestId, &request->m_ALNpOnlineId, &request->m_LanguageCode);
		if(retCheck < 0)
		{
			UNITY_TRACE("ERROR: sceNpGetAccountLanguage() returned %x\n", retCheck);
			Result.SetResultSCE(retCheck, true, __FUNCTION__, __LINE__);
			m_RequestList.erase(request->m_requestId);
			DeleteRequest(request);
		}

		return Result.GetResult();
	}

	ErrorCode NpAsyncRequests::GetParentalControlInfo(const char *onlineId)
	{
		ResultCode Result("");
		if(onlineId ==  NULL)
		{
			UNITY_TRACE("NOTE: GetParentalControlInfo() - missing onlne id\n");
			return NP_ERR_FAILED;
		}


		NpAsyncRequests::UnityNpRequest * request = CreateRequest();
		if(request ==  NULL)
		{
			UNITY_TRACE("NOTE: GetAccountLanguage() - Failed to create asynchronous request\n");
			return NP_ERR_FAILED;
		}
		m_RequestList.insert(std::pair<int,UnityNpRequest *>(request->m_requestId, request) );
		request->m_requestType = UnityNpRequest::TypeParentalControlInfo;

		memcpy(&request->m_PCNpOnlineId , onlineId, min(strlen(onlineId)+1, sizeof(SceNpOnlineId)));
		
		int retCheck = sceNpGetParentalControlInfo(request->m_requestId, &request->m_PCNpOnlineId, &request->m_Age, &request->m_ParentalControlInfo);
		if(retCheck < 0)
		{
			UNITY_TRACE("ERROR: sceNpGetAccountLanguage() returned %x\n", retCheck);
			Result.SetResultSCE(retCheck, true, __FUNCTION__, __LINE__);
			m_RequestList.erase(request->m_requestId);
			DeleteRequest(request);
		}

		return Result.GetResult();
	}
	ErrorCode NpAsyncRequests::CheckNpAvailability(const char *onlineId)
	{
		ResultCode Result("");
		if(onlineId ==  NULL)
		{
			UNITY_TRACE("NOTE: CheckNpAvailability() - missing onlne id\n");
			return NP_ERR_FAILED;
		}

		NpAsyncRequests::UnityNpRequest * request = CreateRequest();
		if(request ==  NULL)
		{
			UNITY_TRACE("NOTE: CheckNpAvailability() - Failed to create asynchronous request\n");
			return NP_ERR_FAILED;
		}

		request->m_requestType = UnityNpRequest::TypeNPAvailabilty;

		memcpy(&request->m_NpOnlineId , onlineId, min(strlen(onlineId)+1, sizeof(SceNpOnlineId)));
		
		int retCheck = sceNpCheckNpAvailability(request->m_requestId, &request->m_NpOnlineId, NULL);
		if(retCheck < 0)
		{
			UNITY_TRACE("ERROR: CheckNpAvailability() returned %x\n", retCheck);
			Result.SetResultSCE(retCheck, true, __FUNCTION__, __LINE__);
		}

		DeleteRequest(request);
		return Result.GetResult();
	}

	ErrorCode NpAsyncRequests::SetGamePresenceOnline(const char *onlineId)
	{
		ResultCode Result("");

		if(onlineId ==  NULL)
		{
			UNITY_TRACE("NOTE: SetGamePresenceOnline() - missing onlne id\n");
			return NP_ERR_FAILED;
		}

		NpAsyncRequests::UnityNpRequest * request = CreateRequest();
		if(request ==  NULL)
		{
			UNITY_TRACE("NOTE: SetGamePresenceOnline() - Failed to create asynchronous request\n");
			return NP_ERR_FAILED;
		}
		m_RequestList.insert(std::pair<int,UnityNpRequest *>(request->m_requestId, request) );
		request->m_requestType = UnityNpRequest::TypeSetGamePresenceOnline;

		memcpy(&request->m_NpOnlineId , onlineId, min(strlen(onlineId)+1, sizeof(SceNpOnlineId)));
		
		int retCheck = sceNpSetGamePresenceOnline(request->m_requestId, &request->m_NpOnlineId, 500, NULL);
		if(retCheck < 0)
		{
			UNITY_TRACE("ERROR: SetGamePresenceOnline() returned %x\n", retCheck);
			Result.SetResultSCE(retCheck, true, __FUNCTION__, __LINE__);
			m_RequestList.erase(request->m_requestId);
			DeleteRequest(request);
		}

		return Result.GetResult();
	}


	// supports overlapping requests
	ErrorCode NpAsyncRequests::CheckPlus(SceUserServiceUserId userId, uint64_t features)
	{
		ResultCode Result("");

		NpAsyncRequests::UnityNpRequest * request = CreateRequest();
		if(request ==  NULL)
		{
			UNITY_TRACE("NOTE: CheckPlus()\n");
			return NP_ERR_FAILED;
		}
		m_RequestList.insert(std::pair<int,UnityNpRequest *>(request->m_requestId, request) );
		request->m_requestType = UnityNpRequest::TypeCheckPlus;

		memset(&request->m_CheckPlusResult, 0, sizeof(request->m_CheckPlusResult));
		memset(&request->m_CheckPlusParameter, 0, sizeof(SceNpCheckPlusParameter));

		request->m_CheckPlusParameter.size = sizeof(SceNpCheckPlusParameter);
		request->m_CheckPlusParameter.userId = userId;
		request->m_CheckPlusParameter.features = features;

		int retCheck = sceNpCheckPlus(request->m_requestId, &request->m_CheckPlusParameter, &request->m_CheckPlusResult);
		if(retCheck < 0)
		{
			switch(retCheck)
			{
				case SCE_NP_ERROR_INVALID_ARGUMENT:							// reqId is not a positive number (>0), or NULL was specified to pOnlineId, pAge or pInfo
				case SCE_NP_ERROR_OUT_OF_MEMORY:							// Not enough free memory
				case SCE_NP_ERROR_SIGNED_OUT:								// Called in the signed out state
				case SCE_NP_ERROR_USER_NOT_FOUND:							// The specified user was not found
				case SCE_NP_ERROR_NOT_SIGNED_UP:							// Called in a non-signed up state
				case SCE_NP_ERROR_AGE_RESTRICTION:							// Applicable to the viewer age restriction
				case SCE_NP_ERROR_LOGOUT:									// Called in a logged out state
				case SCE_NP_ERROR_LATEST_SYSTEM_SOFTWARE_EXIST:				// A new version of the system software update file exists
				case SCE_NP_ERROR_LATEST_SYSTEM_SOFTWARE_EXIST_FOR_TITLE:	// A required new version of the system software update file exists for this application
				case SCE_NP_ERROR_LATEST_PATCH_PKG_EXIST:					// A new version of the update
				case SCE_NP_ERROR_LATEST_PATCH_PKG_DOWNLOADED:				// A new version of the downloaded update file exists
				case SCE_NP_ERROR_ABORTED:									// Processing aborted
				case SCE_NP_ERROR_REQUEST_NOT_FOUND:						// Request specified for reqId does not exist
				default:
					UNITY_TRACE("ERROR: AsyncPlusCheck::Start() - sceNpCheckPlus() returned %x\n", retCheck);
					break;
			}
			m_RequestList.erase(request->m_requestId);
			DeleteRequest(request);

			Result.SetResultSCE(retCheck, true, __FUNCTION__, __LINE__);
		}

		return Result.GetResult();

	}




	// Not actually a request ... but held here for convience
	ErrorCode NpAsyncRequests::NotifyPlusFeature(SceUserServiceUserId userId, uint64_t features)
	{
		SceNpNotifyPlusFeatureParameter notifyParams;

		memset(&notifyParams, 0x0, sizeof(SceNpNotifyPlusFeatureParameter));
		notifyParams.size = sizeof(SceNpNotifyPlusFeatureParameter);
		notifyParams.userId = userId;
		notifyParams.features = features;

		int res = sceNpNotifyPlusFeature(&notifyParams);
		if (res != 0)
		{
			return NP_ERR_FAILED;
		}
		return NP_OK;
	}



#endif

} // namespace UnityPlugin
